const db = require('../config/db');
const { saveBase64Image } = require('../utils/fileStorage');

class CustomerService {
    async getAll() {
        const [rows] = await db.execute('SELECT * FROM customers ORDER BY created_at DESC');
        return rows;
    }

    async getById(id) {
        const [rows] = await db.execute('SELECT * FROM customers WHERE id = ?', [id]);
        return rows[0];
    }

    async create(customerData) {
        let { full_name, phone, email, address, nic, notes, photo_url, nic_front_url, nic_back_url } = customerData;

        // Save images to filesystem
        photo_url = saveBase64Image(photo_url, 'customers/photos');
        nic_front_url = saveBase64Image(nic_front_url, 'customers/nic');
        nic_back_url = saveBase64Image(nic_back_url, 'customers/nic');

        const [result] = await db.execute(
            'INSERT INTO customers (full_name, phone, email, address, nic, notes, photo_url, nic_front_url, nic_back_url) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                full_name || null,
                phone || null,
                email || null,
                address || null,
                nic || null,
                notes || null,
                photo_url || null,
                nic_front_url || null,
                nic_back_url || null
            ]
        );
        return { id: result.insertId, ...customerData, photo_url, nic_front_url, nic_back_url };
    }

    async update(id, customerData) {
        let { full_name, phone, email, address, nic, notes, photo_url, nic_front_url, nic_back_url } = customerData;

        // Ensure no undefined values reach the DB
        email = email || null;
        address = address || null;
        nic = nic || null;
        notes = notes || null;
        photo_url = photo_url || null;
        nic_front_url = nic_front_url || null;
        nic_back_url = nic_back_url || null;

        // Save images to filesystem (only if they are new base64 strings)
        photo_url = saveBase64Image(photo_url, 'customers/photos');
        nic_front_url = saveBase64Image(nic_front_url, 'customers/nic');
        nic_back_url = saveBase64Image(nic_back_url, 'customers/nic');

        await db.execute(
            'UPDATE customers SET full_name = ?, phone = ?, email = ?, address = ?, nic = ?, notes = ?, photo_url = ?, nic_front_url = ?, nic_back_url = ? WHERE id = ?',
            [
                full_name || null,
                phone || null,
                email || null,
                address || null,
                nic || null,
                notes || null,
                photo_url || null,
                nic_front_url || null,
                nic_back_url || null,
                id
            ]
        );
        return { id, ...customerData, photo_url, nic_front_url, nic_back_url };
    }

    async delete(id) {
        await db.execute('DELETE FROM customers WHERE id = ?', [id]);
        return { id, deleted: true };
    }
}

module.exports = new CustomerService();
